/*
 * Decompiled with CFR 0.152.
 */
package ch.epfl.biop.operetta;

import ch.epfl.biop.operetta.utils.HyperRange;
import ij.IJ;
import ij.ImagePlus;
import ij.ImageStack;
import ij.gui.Roi;
import ij.measure.Calibration;
import ij.plugin.HyperStackConverter;
import ij.plugin.ZProjector;
import ij.process.ByteProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import ij.process.LUT;
import ij.process.ShortProcessor;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import loci.formats.FormatException;
import loci.formats.IFormatReader;
import loci.formats.ImageReader;
import loci.formats.Memoizer;
import loci.formats.MetadataTools;
import loci.formats.in.MinimalTiffReader;
import loci.formats.meta.IMetadata;
import loci.formats.meta.MetadataStore;
import mpicbg.models.InvertibleBoundable;
import mpicbg.models.Model;
import mpicbg.models.TranslationModel2D;
import mpicbg.models.TranslationModel3D;
import mpicbg.stitching.CollectionStitchingImgLib;
import mpicbg.stitching.ImageCollectionElement;
import mpicbg.stitching.ImagePlusTimePoint;
import mpicbg.stitching.StitchingParameters;
import mpicbg.stitching.fusion.Fusion;
import net.imglib2.Point;
import net.imglib2.type.numeric.RealType;
import net.imglib2.type.numeric.integer.UnsignedByteType;
import net.imglib2.type.numeric.integer.UnsignedShortType;
import net.imglib2.type.numeric.real.FloatType;
import ome.units.UNITS;
import ome.units.quantity.Length;
import ome.units.quantity.Time;
import ome.xml.meta.MetadataRetrieve;
import ome.xml.meta.OMEXMLMetadataRoot;
import ome.xml.model.Well;
import ome.xml.model.WellSample;
import org.apache.commons.lang.time.StopWatch;
import org.scijava.task.Task;
import org.scijava.task.TaskService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OperettaManager {
    private static final Logger log = LoggerFactory.getLogger(OperettaManager.class);
    private final File id;
    private final IFormatReader main_reader;
    private final IMetadata metadata;
    private final HyperRange range;
    private final double norm_min;
    private final double norm_max;
    private final boolean is_projection;
    private final int projection_type;
    private final File save_folder;
    private final Length px_size;
    private final double correction_factor;
    private final boolean flip_horizontal;
    private final boolean flip_vertical;
    private final int downsample;
    private final boolean fuse_fields;
    private final boolean use_stitcher;
    private StitchingParameters stitching_parameters;
    private final Utilities utils;
    private final TaskService taskService;
    private final boolean use_averaging;

    private OperettaManager(IFormatReader reader, int downsample, boolean use_averaging, HyperRange range, double norm_min, double norm_max, boolean flip_horizontal, boolean flip_vertical, boolean is_projection, int projection_type, File save_folder, double correction_factor, boolean fuse_fields, boolean use_stitcher, StitchingParameters stitching_parameters, TaskService taskService) {
        this.id = new File(reader.getCurrentFile());
        this.main_reader = reader;
        this.metadata = (IMetadata)reader.getMetadataStore();
        this.downsample = downsample;
        this.use_averaging = use_averaging;
        this.range = range;
        this.norm_max = norm_max;
        this.norm_min = norm_min;
        this.flip_horizontal = flip_horizontal;
        this.flip_vertical = flip_vertical;
        this.is_projection = is_projection;
        this.projection_type = projection_type;
        this.save_folder = save_folder;
        this.correction_factor = correction_factor;
        this.fuse_fields = fuse_fields;
        this.use_stitcher = use_stitcher;
        this.stitching_parameters = stitching_parameters;
        this.px_size = this.metadata.getPixelsPhysicalSizeX(0);
        this.utils = new Utilities();
        this.taskService = taskService;
    }

    public String getPlateName() {
        return this.utils.safeName(this.metadata.getPlateName(0));
    }

    public MetadataRetrieve getMetadata() {
        return this.metadata;
    }

    public HyperRange getRange() {
        return this.range;
    }

    public List<Well> getWells() {
        OMEXMLMetadataRoot r = (OMEXMLMetadataRoot)this.metadata.getRoot();
        return r.getPlate(0).copyWellList().stream().filter(well -> ((WellSample)well.copyWellSampleList().get(0)).getPositionX() != null).collect(Collectors.toList());
    }

    public Well getWell(int row, int column) {
        Optional<Well> maybeWell = this.getWells().stream().filter(w -> (Integer)w.getRow().getValue() == row - 1 && (Integer)w.getColumn().getValue() == column - 1).findFirst();
        if (maybeWell.isPresent()) {
            return maybeWell.get();
        }
        log.info("Well at R{}-C{} is not found", (Object)row, (Object)column);
        return null;
    }

    public List<Integer> getFieldIds() {
        int n_fields = this.metadata.getWellSampleCount(0, 0);
        return IntStream.range(1, n_fields + 1).boxed().collect(Collectors.toList());
    }

    public List<WellSample> getFields(Well well) {
        return well.copyWellSampleList();
    }

    public WellSample getField(Well well, int field_id) {
        Optional<WellSample> wellSampleOptional = this.getFields(well).stream().filter(s -> (Integer)s.getIndex().getValue() == field_id).findFirst();
        if (wellSampleOptional.isPresent()) {
            return wellSampleOptional.get();
        }
        log.warn("Field Id " + field_id + "was not found");
        return null;
    }

    public String getWellImageName(Well well) {
        int row = (Integer)well.getRow().getValue() + 1;
        int col = (Integer)well.getColumn().getValue() + 1;
        String project = this.getPlateName();
        String name = String.format("%s - R%02d-C%02d", project, row, col);
        if (this.downsample > 1) {
            name = name + "_Downsample-" + this.downsample;
        }
        if (this.is_projection) {
            name = name + "_Projected-" + ZProjector.METHODS[this.projection_type];
        }
        if (this.use_stitcher) {
            name = name + "_Stitched";
        }
        return name;
    }

    public String getFieldImageName(WellSample field) {
        int row = (Integer)field.getWell().getRow().getValue() + 1;
        int col = (Integer)field.getWell().getColumn().getValue() + 1;
        String field_id = field.getID();
        String local_field_id = field_id.substring(field_id.lastIndexOf(":") + 1);
        String project = this.utils.safeName(field.getWell().getPlate().getName());
        String name = String.format("%s - R%02d-C%02d-F%s", project, row, col, local_field_id);
        if (this.downsample > 1) {
            name = name + "_Downsample-" + this.downsample;
        }
        if (this.is_projection) {
            name = name + "_Projected-" + ZProjector.METHODS[this.projection_type];
        }
        return name;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ImagePlus getStitchedWellImage(Well well, List<WellSample> fields, Roi subregion) {
        String imageName = this.getWellImageName(well);
        File stitchingfFile = new File(this.save_folder, imageName + ".txt");
        if (fields == null) {
            fields = this.getFields(well);
        }
        if (subregion != null) {
            fields = this.getIntersectingFields(fields, subregion);
        }
        AtomicInteger index = new AtomicInteger(0);
        ArrayList stitchingElements = new ArrayList();
        fields.forEach(f -> {
            float[] fArray;
            int dimensionality;
            ImagePlus field = this.getFieldImage((WellSample)f);
            Point positionXY = this.utils.getUncalibratedCoordinates((WellSample)f);
            int n = dimensionality = field.getNSlices() > 1 ? 3 : 2;
            if (dimensionality > 2) {
                float[] fArray2 = new float[3];
                fArray2[0] = 0.0f;
                fArray2[1] = 0.0f;
                fArray = fArray2;
                fArray2[2] = 0.0f;
            } else {
                float[] fArray3 = new float[2];
                fArray3[0] = 0.0f;
                fArray = fArray3;
                fArray3[1] = 0.0f;
            }
            float[] offset = fArray;
            if (positionXY != null) {
                offset[0] = (float)positionXY.getLongPosition(0) / (float)this.downsample;
                offset[1] = (float)positionXY.getLongPosition(1) / (float)this.downsample;
            }
            ImageCollectionElement element = new ImageCollectionElement(stitchingfFile, index.getAndIncrement());
            element.setOffset(offset);
            element.setDimensionality(dimensionality);
            element.setImagePlus(field);
            element.setModel((Model)(dimensionality == 2 ? new TranslationModel2D() : new TranslationModel3D()));
            stitchingElements.add(element);
        });
        if (this.stitching_parameters == null) {
            this.stitching_parameters = new StitchingParameters();
            this.stitching_parameters.channel1 = 0;
            this.stitching_parameters.channel2 = 0;
            this.stitching_parameters.timeSelect = 0;
            this.stitching_parameters.checkPeaks = 5;
            this.stitching_parameters.fusionMethod = 0;
            this.stitching_parameters.regThreshold = 0.3;
            this.stitching_parameters.relativeThreshold = 2.5;
            this.stitching_parameters.absoluteThreshold = 3.5;
            this.stitching_parameters.dimensionality = ((ImageCollectionElement)stitchingElements.get(0)).getDimensionality();
        }
        int size = stitchingElements.size();
        ArrayList<ImagePlus> images = new ArrayList<ImagePlus>(size);
        ArrayList<InvertibleBoundable> models = new ArrayList<InvertibleBoundable>(size);
        ArrayList optimized = CollectionStitchingImgLib.stitchCollection(stitchingElements, (StitchingParameters)this.stitching_parameters);
        for (ImagePlusTimePoint imagePlusTimePoint : optimized) {
            images.add(imagePlusTimePoint.getImagePlus());
            models.add((InvertibleBoundable)imagePlusTimePoint.getModel());
        }
        try {
            ImagePlus fused;
            switch (((ImagePlus)images.get(0)).getBitDepth()) {
                case 8: {
                    UnsignedByteType fusedImageByte = new UnsignedByteType();
                    fused = Fusion.fuse((RealType)fusedImageByte, images, models, (int)((ImageCollectionElement)stitchingElements.get(0)).getDimensionality(), (boolean)false, (int)0, null, (boolean)false, (boolean)false, (boolean)false);
                    break;
                }
                case 32: {
                    FloatType fusedImageFloat = new FloatType();
                    fused = Fusion.fuse((RealType)fusedImageFloat, images, models, (int)((ImageCollectionElement)stitchingElements.get(0)).getDimensionality(), (boolean)false, (int)0, null, (boolean)false, (boolean)false, (boolean)false);
                    break;
                }
                default: {
                    UnsignedShortType fusedImageShort = new UnsignedShortType();
                    fused = Fusion.fuse((RealType)fusedImageShort, images, models, (int)((ImageCollectionElement)stitchingElements.get(0)).getDimensionality(), (boolean)false, (int)0, null, (boolean)false, (boolean)false, (boolean)false);
                }
            }
            fused.setTitle(imageName);
            ImagePlus ref = (ImagePlus)images.get(0);
            for (int s = 0; s < ref.getStackSize(); ++s) {
                LUT lut = ref.getStack().getProcessor(s + 1).getLut();
                fused.getStack().getProcessor(s + 1).setLut(lut);
            }
            fused.setCalibration(ref.getCalibration());
            fused.setProperties(ref.getPropertiesAsArray());
            fused.setProperty("Stitching", (Object)"Using OperettaImporter");
            for (int c = 0; c < ref.getNChannels(); ++c) {
                fused.setPosition(c + 1, 1, 1);
                fused.resetDisplayRange();
            }
            ImagePlus imagePlus = fused;
            return imagePlus;
        }
        finally {
            images.forEach(ImagePlus::close);
        }
    }

    public ImagePlus getWellImage(Well well) {
        return this.getWellImage(well, null, null);
    }

    public ImagePlus getWellImage(Well well, List<WellSample> fields, Roi bounds) {
        List<WellSample> adjusted_fields;
        if (this.use_stitcher) {
            return this.getStitchedWellImage(well, fields, bounds);
        }
        if (fields == null) {
            fields = well.copyWellSampleList();
        }
        if ((adjusted_fields = this.getIntersectingFields(fields, bounds)).isEmpty()) {
            return null;
        }
        int a_field_id = (Integer)((WellSample)fields.get(0)).getIndex().getValue();
        int sample_width = (Integer)this.metadata.getPixelsSizeX(a_field_id).getValue();
        int sample_height = (Integer)this.metadata.getPixelsSizeY(a_field_id).getValue();
        Point topLeftCoordinates = this.utils.getTopLeftCoordinates(fields);
        Point bottomRightCoordinates = this.utils.getBottomRightCoordinates(fields);
        if (topLeftCoordinates == null || bottomRightCoordinates == null) {
            log.error("Could not find coordinates for well " + well);
            return null;
        }
        long well_width = bottomRightCoordinates.getLongPosition(0) - topLeftCoordinates.getLongPosition(0) + (long)sample_width;
        long well_height = bottomRightCoordinates.getLongPosition(1) - topLeftCoordinates.getLongPosition(1) + (long)sample_height;
        if (bounds != null) {
            well_width = bounds.getBounds().width;
            well_height = bounds.getBounds().height;
        }
        HyperRange range2 = this.range.confirmRange(this.metadata);
        int n = range2.getTotalPlanes();
        ImageStack wellStack = ImageStack.create((int)((int)(well_width /= (long)this.downsample)), (int)((int)(well_height /= (long)this.downsample)), (int)n, (int)16);
        AtomicInteger ai = new AtomicInteger(0);
        adjusted_fields.forEach(field -> {
            Roi subregion = this.getFieldSubregion((WellSample)field, bounds, topLeftCoordinates);
            int field_counter = ai.getAndIncrement();
            if (subregion != null) {
                Point pos = this.utils.getFieldAdjustedCoordinates((WellSample)field, bounds, subregion, topLeftCoordinates);
                log.info(String.format("Sample Position: %d, %d", pos.getLongPosition(0), pos.getLongPosition(1)));
                ImageStack stack = this.getFieldImage((WellSample)field, subregion).getStack();
                if (stack != null) {
                    for (int s = 0; s < stack.size(); ++s) {
                        wellStack.getProcessor(s + 1).copyBits(stack.getProcessor(s + 1), (int)pos.getLongPosition(0), (int)pos.getLongPosition(1), 0);
                        wellStack.setSliceLabel(stack.getSliceLabel(s + 1), s + 1);
                    }
                    log.info(String.format("Field %d of %d Copied to Well", field_counter + 1, adjusted_fields.size()));
                }
            } else {
                log.warn(String.format("Field %d of %d not found.", field_counter + 1, adjusted_fields.size()));
            }
        });
        if (wellStack == null) {
            return null;
        }
        int[] czt = this.range.getCZTDimensions();
        String imageName = this.getWellImageName(well);
        ImagePlus result = new ImagePlus(imageName, wellStack);
        if (czt[0] + czt[1] + czt[2] > 3) {
            result = HyperStackConverter.toHyperStack((ImagePlus)result, (int)czt[0], (int)czt[1], (int)czt[2]);
        }
        Calibration cal = new Calibration(result);
        Calibration meta = this.utils.getCalibration();
        cal.pixelWidth = meta.pixelWidth;
        cal.pixelHeight = meta.pixelHeight;
        cal.pixelDepth = meta.pixelDepth;
        cal.frameInterval = meta.frameInterval;
        cal.setXUnit(meta.getXUnit());
        cal.setYUnit(meta.getYUnit());
        cal.setZUnit(meta.getZUnit());
        cal.setTimeUnit(meta.getTimeUnit());
        if (this.is_projection && result.getNSlices() > 1) {
            ZProjector zp = new ZProjector();
            zp.setImage(result);
            zp.setMethod(this.projection_type);
            zp.setStopSlice(result.getNSlices());
            if (result.getNSlices() > 1 || result.getNFrames() > 1) {
                zp.doHyperStackProjection(true);
            }
            result = zp.getProjection();
        }
        Point point = this.utils.getTopLeftCoordinatesUm(fields);
        cal.xOrigin = point.getDoublePosition(0) / cal.pixelWidth;
        cal.yOrigin = point.getDoublePosition(1) / cal.pixelHeight;
        result.setCalibration(cal);
        return result;
    }

    public ImagePlus getFieldImage(WellSample field) {
        return this.getFieldImage(field, null);
    }

    public ImagePlus getFieldImage(WellSample field, Roi subregion) {
        int series_id = (Integer)field.getIndex().getValue();
        int row = (Integer)field.getWell().getRow().getValue();
        int column = (Integer)field.getWell().getColumn().getValue();
        this.main_reader.setSeries(series_id);
        HyperRange range2 = this.range.confirmRange(this.metadata);
        int n = range2.getTotalPlanes();
        boolean do_norm = this.main_reader.getBitsPerPixel() != 16;
        int stack_width = this.main_reader.getSizeX();
        int stack_height = this.main_reader.getSizeY();
        if (subregion != null) {
            stack_width = subregion.getBounds().width;
            stack_height = subregion.getBounds().height;
        }
        if ((stack_height /= this.downsample) <= 1 || (stack_width /= this.downsample) <= 1) {
            return null;
        }
        ImageStack stack = ImageStack.create((int)stack_width, (int)stack_height, (int)n, (int)16);
        List files = Arrays.stream(this.main_reader.getSeriesUsedFiles(false)).filter(f -> f.endsWith(".tiff")).collect(Collectors.toList());
        org.perf4j.StopWatch sw = new org.perf4j.StopWatch();
        sw.start();
        ForkJoinPool planeWorkerPool = new ForkJoinPool(10);
        try {
            ((ForkJoinTask)planeWorkerPool.submit(() -> IntStream.range(0, files.size()).parallel().forEach(i -> {
                Map<String, Integer> plane_indexes = range2.getIndexes((String)files.get(i));
                if (range2.includes((String)files.get(i))) {
                    ImageProcessor ip = this.openTiffFileAsImageProcessor((String)files.get(i));
                    if (ip == null) {
                        log.error("Could not open {}", files.get(i));
                    } else {
                        if (this.flip_horizontal) {
                            ip.flipHorizontal();
                        }
                        if (this.flip_vertical) {
                            ip.flipVertical();
                        }
                        if (do_norm) {
                            ip.setMinAndMax(this.norm_min, this.norm_max);
                            ip = ip.convertToShort(true);
                        }
                        if (subregion != null) {
                            ip.setRoi(subregion);
                            ip = ip.crop();
                        }
                        ip = ip.resize(ip.getWidth() / this.downsample, ip.getHeight() / this.downsample, this.use_averaging);
                        String label = String.format("R%d-C%d - (c:%d, z:%d, t:%d) - %s", row, column, plane_indexes.get("C"), plane_indexes.get("Z"), plane_indexes.get("T"), new File((String)files.get(i)).getName());
                        stack.setProcessor(ip, plane_indexes.get("I").intValue());
                        stack.setSliceLabel(label, plane_indexes.get("I").intValue());
                    }
                }
            }))).get();
        }
        catch (InterruptedException e) {
            log.error("Reading Stack " + series_id + " interrupted:", e);
        }
        catch (ExecutionException e) {
            log.error("Reading Stack " + series_id + " error:", e);
        }
        if (stack == null) {
            return null;
        }
        int[] czt = this.range.getCZTDimensions();
        String imageName = this.getFieldImageName(field);
        ImagePlus result = new ImagePlus(imageName, stack);
        if (czt[0] + czt[1] + czt[2] > 3) {
            result = HyperStackConverter.toHyperStack((ImagePlus)result, (int)czt[0], (int)czt[1], (int)czt[2]);
        }
        Calibration cal = new Calibration(result);
        Calibration meta = this.utils.getCalibration();
        cal.pixelWidth = meta.pixelWidth;
        cal.pixelHeight = meta.pixelHeight;
        cal.pixelDepth = meta.pixelDepth;
        cal.frameInterval = meta.frameInterval;
        cal.setXUnit(meta.getXUnit());
        cal.setYUnit(meta.getYUnit());
        cal.setZUnit(meta.getZUnit());
        cal.setTimeUnit(meta.getTimeUnit());
        if (this.is_projection && result.getNSlices() > 1) {
            ZProjector zp = new ZProjector();
            zp.setImage(result);
            zp.setMethod(this.projection_type);
            zp.setStopSlice(result.getNSlices());
            if (result.getNSlices() > 1 || result.getNFrames() > 1) {
                zp.doHyperStackProjection(true);
            }
            result = zp.getProjection();
        }
        Point point = this.utils.getTopLeftCoordinatesUm(Collections.singletonList(field));
        cal.xOrigin = point.getDoublePosition(0) / cal.pixelWidth;
        cal.yOrigin = point.getDoublePosition(1) / cal.pixelHeight;
        result.setCalibration(cal);
        sw.stop();
        log.info("Field " + field.getID() + " stack " + series_id + " took " + (double)sw.getElapsedTime() / 1000.0 + " seconds");
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private ImageProcessor openTiffFileAsImageProcessor(String id) {
        try (MinimalTiffReader reader = new MinimalTiffReader();){
            reader.setId(id);
            reader.setSeries(0);
            int width = reader.getSizeX();
            int height = reader.getSizeY();
            byte[] bytes = reader.openBytes(0);
            ByteOrder byteOrder = reader.isLittleEndian() ? ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN;
            switch (reader.getPixelType()) {
                case 1: {
                    ByteProcessor byteProcessor = new ByteProcessor(width, height, bytes, null);
                    return byteProcessor;
                }
                case 3: {
                    ByteBuffer buffer = ByteBuffer.allocate(width * height * 2);
                    buffer.put(bytes);
                    short[] shorts = new short[width * height];
                    buffer.flip();
                    buffer.order(byteOrder).asShortBuffer().get(shorts);
                    ShortProcessor shortProcessor = new ShortProcessor(width, height, shorts, null);
                    return shortProcessor;
                }
                case 6: {
                    ByteBuffer forFloatBuffer = ByteBuffer.allocate(width * height * 4);
                    forFloatBuffer.put(bytes);
                    float[] floats = new float[width * height];
                    forFloatBuffer.flip();
                    forFloatBuffer.order(byteOrder).asFloatBuffer().get(floats);
                    FloatProcessor floatProcessor = new FloatProcessor(width, height, floats, null);
                    return floatProcessor;
                }
            }
            log.error("No idea what the data type is for image " + id + " : " + reader.getPixelType());
            ImageProcessor imageProcessor = null;
            return imageProcessor;
        }
        catch (IOException | FormatException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public void process() {
        this.process(null, null, null);
    }

    public void process(List<Well> wells) {
        this.process(wells, null, null);
    }

    public void process(Roi region) {
        this.process(null, null, region);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void process(List<Well> wells, List<Integer> fields, Roi region) {
        if (wells == null) {
            wells = this.getWells();
        }
        AtomicInteger iWell = new AtomicInteger();
        Instant global_start = Instant.now();
        Task taskWell = null;
        Task taskField = null;
        if (this.taskService != null) {
            taskWell = this.taskService.createTask("Export of " + wells.size() + " well(s)");
            taskWell.setProgressMaximum((long)wells.size());
            taskWell.start();
        }
        try {
            for (Well well : wells) {
                double percentageCompleteness;
                if (taskWell != null) {
                    if (taskWell.isCanceled()) {
                        IJ.log((String)("The export task has been cancelled: " + taskWell.getCancelReason()));
                        return;
                    }
                    taskWell.setStatusMessage("- Well " + well.getID());
                }
                log.info("Well: {}", (Object)well);
                IJ.log((String)("- Well " + well.getID() + " (" + iWell + "/" + wells.size() + " )"));
                Instant well_start = Instant.now();
                List<WellSample> well_fields = fields != null ? fields.stream().map(arg_0 -> ((Well)well).getWellSample(arg_0)).collect(Collectors.toList()) : well.copyWellSampleList();
                if (!this.fuse_fields) {
                    if (region != null) {
                        well_fields = this.getIntersectingFields(well_fields, region);
                    }
                    if (this.taskService != null) {
                        taskField = this.taskService.createTask("Export " + well_fields.size() + " Fields");
                        taskField.start();
                        taskField.setProgressMaximum((long)well_fields.size());
                    }
                    AtomicInteger iField = new AtomicInteger();
                    try {
                        for (WellSample field : well_fields) {
                            if (taskField != null) {
                                if (taskField.isCanceled()) {
                                    assert (taskWell != null);
                                    taskWell.cancel("Downstream task cancelled.");
                                    break;
                                }
                                taskField.setStatusMessage("- " + field.getID());
                            }
                            iField.incrementAndGet();
                            IJ.log((String)("\t - Field " + field.getID() + " (" + iField + "/" + well_fields.size() + ")"));
                            ImagePlus field_image = this.getFieldImage(field, null);
                            String name = this.getFieldImageName(field);
                            if (field_image != null) {
                                IJ.saveAsTiff((ImagePlus)field_image, (String)new File(this.save_folder, name + ".tif").getAbsolutePath());
                            }
                            percentageCompleteness = ((double)iWell.get() / (double)wells.size() + (double)iField.get() / (double)(well_fields.size() * wells.size())) * 100.0;
                            this.utils.printTimingMessage(global_start, percentageCompleteness);
                            if (taskField == null) continue;
                            taskField.setProgressValue((long)iField.get());
                        }
                        try {
                            this.writeWellPositionsFile(well_fields, new File(this.save_folder, this.getWellImageName(well) + ".txt"));
                        }
                        catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                    finally {
                        if (taskField != null) {
                            taskField.finish();
                        }
                    }
                } else {
                    ImagePlus well_image = this.getWellImage(well, well_fields, region);
                    String name = this.getWellImageName(well);
                    if (well_image != null) {
                        IJ.saveAsTiff((ImagePlus)well_image, (String)new File(this.save_folder, name + ".tif").getAbsolutePath());
                    }
                }
                Instant ends = Instant.now();
                IJ.log((String)(" - Well processed in " + Duration.between(well_start, ends).getSeconds() + " s."));
                iWell.incrementAndGet();
                if (taskWell != null) {
                    taskWell.setProgressValue((long)iWell.get());
                }
                percentageCompleteness = (double)iWell.get() / (double)wells.size() * 100.0;
                this.utils.printTimingMessage(global_start, percentageCompleteness);
            }
            Instant global_ends = Instant.now();
            IJ.log((String)(" DONE! All wells processed in " + Duration.between(global_start, global_ends).getSeconds() / 60L + " min."));
        }
        finally {
            if (taskWell != null) {
                taskWell.finish();
            }
            if (taskField != null) {
                taskField.finish();
            }
        }
    }

    public String toString() {
        String expName;
        int nWells = this.getWells().size();
        int nFields = this.getFieldIds().size();
        long[] dims = this.utils.getIODimensions();
        try {
            expName = this.getPlateName();
        }
        catch (IndexOutOfBoundsException e) {
            expName = "Unknown";
        }
        String dataInfo = String.format("Operetta plate '%s' contains: \n\t %d Wells, each containing \n\t %d Fields, with dimensions \n\t (X, Y, Z, C, T) = (%d, %d, %d, %d, %d)\n\tFile location: '%s'", expName, nWells, nFields, dims[0], dims[1], dims[2], dims[3], dims[4], this.id.getParentFile().getAbsolutePath());
        dataInfo = dataInfo + "\n\n Operetta Manager parameters:\n";
        dataInfo = dataInfo + String.format("- Downsample factor: %d\n", this.downsample);
        dataInfo = dataInfo + String.format("\t- Use averaging when downsampling: %b\n", this.use_averaging);
        dataInfo = dataInfo + String.format("- Fusing fields: %b\n- Use Grid/Collection Stitching for fusion: %b\n\n", this.fuse_fields, this.use_stitcher);
        dataInfo = dataInfo + String.format("- Tile position correction factor: %.2f\n\t- Horizontal camera flip: %b\n\t- Vertical camera flip: %b\n\t", this.correction_factor, this.flip_horizontal, this.flip_vertical);
        dataInfo = dataInfo + String.format("- 32-bit Digital Phase Contrast image normalization\n\t\t- Min: %.2f\n\t\t- Max: %.2f\n\n", this.norm_min, this.norm_max);
        dataInfo = dataInfo + String.format("- Performing projection: %b\n\t- Projection type: %s\n\n", this.is_projection, ZProjector.METHODS[this.projection_type]);
        dataInfo = dataInfo + String.format("- Selected ranges:\n\t- Z: %s", HyperRange.prettyPrint(this.range.getRangeZ()));
        dataInfo = dataInfo + String.format("\n\t- C: %s", HyperRange.prettyPrint(this.range.getRangeC()));
        dataInfo = dataInfo + String.format("\n\t- T: %s", HyperRange.prettyPrint(this.range.getRangeT()));
        return dataInfo;
    }

    public void writeWellPositionsFile(List<WellSample> samples, File position_file) throws IOException {
        int dim = this.range.getRangeZ().size() > 1 && !this.is_projection ? 3 : 2;
        String z = dim == 3 ? ", 0.0" : "";
        Path path = Paths.get(position_file.getAbsolutePath(), new String[0]);
        try (BufferedWriter writer = Files.newBufferedWriter(path, new OpenOption[0]);){
            writer.write("#Define the number of dimensions we are working on:\n");
            writer.write("dim = " + dim + "\n");
            writer.write("# Define the image coordinates\n");
            writer.write("#Define the number of dimensions we are working on:\n");
            for (WellSample sample : samples) {
                String name = this.getFieldImageName(sample);
                Point pos = this.utils.getUncalibratedCoordinates(sample);
                if (pos != null) {
                    writer.write(String.format("%s.tif;      ;               (%d.0, %d.0%s)\n", name, pos.getLongPosition(0) / (long)this.downsample, pos.getLongPosition(1) / (long)this.downsample, z));
                    continue;
                }
                writer.write(String.format("# %s.tif;      ;               (NaN, NaN%s)\n", name, z));
            }
        }
    }

    public List<WellSample> getIntersectingFields(List<WellSample> fields, Roi bounds) {
        if (bounds == null) {
            return fields;
        }
        log.info("Looking for samples intersecting with {}, ", (Object)bounds);
        Point topLeftCoordinates = this.utils.getTopLeftCoordinates(fields);
        if (topLeftCoordinates == null) {
            log.error("No coordinates found for fields " + fields.toString() + " -> returning all fields.");
            return fields;
        }
        List<WellSample> selected = fields.stream().filter(s -> {
            int sample_id = (Integer)s.getIndex().getValue();
            Long pX = this.utils.getUncalibratedPositionX(s);
            if (pX == null) {
                return false;
            }
            Long pY = this.utils.getUncalibratedPositionY(s);
            if (pY == null) {
                return false;
            }
            long x = pX - topLeftCoordinates.getLongPosition(0);
            long y = pY - topLeftCoordinates.getLongPosition(1);
            int w = (Integer)this.metadata.getPixelsSizeX(sample_id).getValue();
            int h = (Integer)this.metadata.getPixelsSizeY(sample_id).getValue();
            Roi other = new Roi((double)x, (double)y, (double)w, (double)h);
            return this.utils.isOverlapping(bounds, other);
        }).collect(Collectors.toList());
        IJ.log((String)("Selected Samples: " + selected));
        return selected;
    }

    private Roi getFieldSubregion(WellSample field, Roi bounds, Point topLeftCoordinates) {
        long x = 0L;
        long y = 0L;
        int sample_id = (Integer)field.getIndex().getValue();
        if (this.metadata.getPixelsSizeX(sample_id) == null || this.metadata.getPixelsSizeY(sample_id) == null) {
            return null;
        }
        long w = ((Integer)this.metadata.getPixelsSizeX(sample_id).getValue()).intValue();
        long h = ((Integer)this.metadata.getPixelsSizeY(sample_id).getValue()).intValue();
        Point coordinates = this.utils.getUncalibratedCoordinates(field);
        if (coordinates == null) {
            return null;
        }
        coordinates.move(new long[]{-topLeftCoordinates.getLongPosition(0), -topLeftCoordinates.getLongPosition(1)});
        if (bounds != null) {
            if ((long)bounds.getBounds().x > coordinates.getLongPosition(0)) {
                x = (long)bounds.getBounds().x - coordinates.getLongPosition(0);
                w -= x;
            }
            if ((long)bounds.getBounds().y > coordinates.getLongPosition(1)) {
                y = (long)bounds.getBounds().y - coordinates.getLongPosition(1);
                h -= y;
            }
        }
        return new Roi((double)x, (double)y, (double)w, (double)h);
    }

    public Utilities getUtilities() {
        return this.utils;
    }

    public class Utilities {
        private String safeName(String nameUnsafe) {
            return nameUnsafe.replaceAll("[^\\w\\s\\-_]", "_");
        }

        private void printTimingMessage(Instant start, double percentageCompleteness) {
            long s = Duration.between(start, Instant.now()).getSeconds();
            String elapsedTime = String.format("%d:%02d:%02d", s / 3600L, s % 3600L / 60L, s % 60L);
            double sPerPC = (double)s / percentageCompleteness;
            long sRemaining = (long)((100.0 - percentageCompleteness) * sPerPC);
            String remainingTime = String.format("%d:%02d:%02d", sRemaining / 3600L, sRemaining % 3600L / 60L, sRemaining % 60L);
            LocalDateTime estimateDoneJob = LocalDateTime.now().plus(Duration.ofSeconds(sRemaining));
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
            long nDays = sRemaining / 86400L;
            String daysMessage = "";
            if (nDays == 1L) {
                daysMessage = daysMessage + " tomorrow.";
            }
            if (nDays == 1L) {
                daysMessage = daysMessage + " in " + nDays + " days.";
            }
            String formatDateTime = estimateDoneJob.format(formatter);
            IJ.log((String)(" -  Task " + (int)percentageCompleteness + " % completed. Elapsed time:" + elapsedTime + ". Estimated remaining time: " + remainingTime + ". Job done at around " + formatDateTime + daysMessage));
        }

        public Calibration getCalibration() {
            Length apx_depth;
            int[] czt = OperettaManager.this.range.getCZTDimensions();
            double px_size = 1.0;
            double px_depth = 1.0;
            String v_unit = "pixel";
            double px_time = 1.0;
            String time_unit = "sec";
            Length apx_size = OperettaManager.this.metadata.getPixelsPhysicalSizeX(0);
            if (apx_size != null) {
                px_size = apx_size.value(UNITS.MICROMETER).doubleValue();
                v_unit = UNITS.MICROMETER.getSymbol();
            }
            if ((apx_depth = OperettaManager.this.metadata.getPixelsPhysicalSizeZ(0)) != null) {
                px_depth = apx_depth.value(UNITS.MICROMETER).doubleValue();
            }
            if (OperettaManager.this.range.getCZTDimensions()[2] > 1) {
                Time apx_time;
                Time time = apx_time = OperettaManager.this.metadata.getPixelsTimeIncrement(0) == null ? OperettaManager.this.metadata.getPlaneDeltaT(0, czt[0] * czt[1]) : null;
                if (apx_time != null) {
                    px_time = apx_time.value(UNITS.SECOND).doubleValue();
                    time_unit = UNITS.SECOND.getSymbol();
                }
            }
            Calibration cal = new Calibration();
            cal.pixelWidth = px_size;
            cal.pixelHeight = px_size;
            cal.pixelDepth = px_depth;
            cal.frameInterval = px_time;
            cal.setXUnit(v_unit);
            cal.setYUnit(v_unit);
            cal.setZUnit(v_unit);
            cal.setTimeUnit(time_unit);
            return cal;
        }

        private Long getUncalibratedPositionX(WellSample field) {
            Length px = field.getPositionX();
            if (px == null) {
                return null;
            }
            double px_m = px.value(UNITS.MICROMETER).doubleValue();
            return Math.round(px_m / OperettaManager.this.px_size.value(UNITS.MICROMETER).doubleValue() * OperettaManager.this.correction_factor);
        }

        private Long getUncalibratedPositionY(WellSample field) {
            Length px = field.getPositionY();
            if (px == null) {
                return null;
            }
            double px_m = px.value(UNITS.MICROMETER).doubleValue();
            return Math.round(px_m / OperettaManager.this.px_size.value(UNITS.MICROMETER).doubleValue() * OperettaManager.this.correction_factor);
        }

        public Point getUncalibratedCoordinates(WellSample field) {
            Long px = this.getUncalibratedPositionX(field);
            Long py = this.getUncalibratedPositionY(field);
            if (px == null || py == null) {
                return null;
            }
            return new Point(new long[]{px, py});
        }

        public Point getFieldAdjustedCoordinates(WellSample field, Roi bounds, Roi subregion, Point topLeftCoordinates) {
            Point pos = this.getUncalibratedCoordinates(field);
            if (pos == null) {
                log.error("Could not find position for field " + field);
                throw new RuntimeException("Could not find position for field " + field);
            }
            pos.move(new long[]{-topLeftCoordinates.getLongPosition(0), -topLeftCoordinates.getLongPosition(1)});
            if (bounds != null) {
                pos.move(new long[]{subregion.getBounds().x - bounds.getBounds().x, subregion.getBounds().y - bounds.getBounds().y});
            }
            pos.setPosition(new long[]{pos.getLongPosition(0) / (long)OperettaManager.this.downsample, pos.getLongPosition(1) / (long)OperettaManager.this.downsample});
            return pos;
        }

        public Point getTopLeftCoordinates(List<WellSample> fields) {
            if ((fields = fields.stream().filter(sample -> sample.getPositionX() != null).collect(Collectors.toList())).isEmpty()) {
                System.err.println("Cannot find coordinates");
                return null;
            }
            Optional<WellSample> minx = fields.stream().min(Comparator.comparing(WellSample::getPositionX));
            Optional<WellSample> miny = fields.stream().min(Comparator.comparing(WellSample::getPositionY));
            if (!minx.isPresent() || !miny.isPresent()) {
                log.info("Could not find top left coordinates for fields");
                return null;
            }
            Long px = this.getUncalibratedPositionX(minx.get());
            Long py = this.getUncalibratedPositionY(miny.get());
            return new Point(new long[]{px, py});
        }

        public Point getTopLeftCoordinatesUm(List<WellSample> fields) {
            fields = fields.stream().filter(sample -> sample.getPositionX() != null).filter(sample -> sample.getPositionX().value() != null).collect(Collectors.toList());
            Optional<WellSample> minx = fields.stream().min(Comparator.comparing(WellSample::getPositionX));
            Optional<WellSample> miny = fields.stream().min(Comparator.comparing(WellSample::getPositionY));
            if (!minx.isPresent() || !miny.isPresent()) {
                log.info("Could not find top left coordinates for fields");
                return new Point(new int[]{0, 0});
            }
            Long px = minx.get().getPositionX().value(UNITS.MICROMETER).longValue();
            Long py = miny.get().getPositionY().value(UNITS.MICROMETER).longValue();
            return new Point(new long[]{px, py});
        }

        public Point getBottomRightCoordinates(List<WellSample> fields) {
            if (!(fields = fields.stream().filter(sample -> sample.getPositionY() != null).collect(Collectors.toList())).isEmpty()) {
                Optional<WellSample> maxX = fields.stream().max(Comparator.comparing(WellSample::getPositionX));
                Optional<WellSample> maxY = fields.stream().max(Comparator.comparing(WellSample::getPositionY));
                if (!maxX.isPresent() || !maxY.isPresent()) {
                    log.info("Could not find top left coordinates for fields");
                    return null;
                }
                Long px = this.getUncalibratedPositionX(maxX.get());
                Long py = this.getUncalibratedPositionY(maxY.get());
                if (px != null && py != null) {
                    return new Point(new long[]{px, py});
                }
                return null;
            }
            System.err.println("All fields are uncalibrated!");
            return null;
        }

        private Roi resampleRoi(Roi r, int s) {
            return new Roi(r.getBounds().x / s, r.getBounds().y / s, r.getBounds().width / s, r.getBounds().height / s);
        }

        private boolean isOverlapping(Roi one, Roi other) {
            return one.getBounds().intersects(other.getBounds());
        }

        public long[] getIOBytes(List<Well> wells, List<Integer> fields) {
            long nWells = wells.size();
            long nFields = fields.size();
            long nTotalPlanes = OperettaManager.this.range.getTotalPlanes();
            long sX = OperettaManager.this.main_reader.getSizeX();
            long sY = OperettaManager.this.main_reader.getSizeY();
            long nFieldsOut = nFields;
            long sXOut = OperettaManager.this.main_reader.getSizeX() / OperettaManager.this.downsample;
            long sYOut = OperettaManager.this.main_reader.getSizeY() / OperettaManager.this.downsample;
            long sZOut = OperettaManager.this.is_projection ? 1L : (long)OperettaManager.this.getRange().getRangeZ().size();
            long sCOut = OperettaManager.this.getRange().getRangeC().size();
            long sTOut = OperettaManager.this.getRange().getRangeT().size();
            return new long[]{sX * sY * nTotalPlanes * nFields * nWells * 2L, sXOut * sYOut * sZOut * sCOut * sTOut * nFieldsOut * nWells * 2L};
        }

        public long[] getIODimensions() {
            long sX = OperettaManager.this.main_reader.getSizeX();
            long sY = OperettaManager.this.main_reader.getSizeY();
            long sZ = OperettaManager.this.main_reader.getSizeZ();
            long sC = OperettaManager.this.main_reader.getSizeC();
            long sT = OperettaManager.this.main_reader.getSizeT();
            long sXOut = OperettaManager.this.main_reader.getSizeX() / OperettaManager.this.downsample;
            long sYOut = OperettaManager.this.main_reader.getSizeY() / OperettaManager.this.downsample;
            long sZOut = OperettaManager.this.is_projection ? 1L : (long)OperettaManager.this.getRange().getRangeZ().size();
            long sCOut = OperettaManager.this.getRange().getRangeC().size();
            long sTOut = OperettaManager.this.getRange().getRangeT().size();
            return new long[]{sX, sY, sZ, sC, sT, sXOut, sYOut, sZOut, sCOut, sTOut};
        }
    }

    public static class Builder {
        private File id = null;
        private double norm_min = 0.0;
        private double norm_max = Math.pow(2.0, 16.0);
        private HyperRange range = null;
        private double correction_factor = 0.995;
        private boolean is_projection = false;
        private int projection_method = 1;
        private File save_folder = new File(System.getProperty("user.home"));
        private IFormatReader reader = null;
        private boolean flip_horizontal = false;
        private boolean flip_vertical = false;
        private TaskService taskService = null;
        private int downsample = 1;
        private boolean is_fuse_fields = false;
        private boolean is_use_stitcher = false;
        private StitchingParameters stitching_parameters = null;
        private boolean use_averaging = false;

        public Builder setStitchingParameters(StitchingParameters stitching_parameters) {
            this.stitching_parameters = stitching_parameters;
            this.is_use_stitcher = true;
            this.is_fuse_fields = true;
            return this;
        }

        public Builder fuseFields(boolean is_fuse_fields) {
            this.is_fuse_fields = is_fuse_fields;
            if (!is_fuse_fields) {
                this.is_use_stitcher = false;
            }
            return this;
        }

        public Builder useStitcher(boolean is_use_stitcher) {
            this.is_use_stitcher = is_use_stitcher;
            if (is_use_stitcher) {
                this.is_fuse_fields = true;
            }
            return this;
        }

        public Builder flipHorizontal(boolean isFlip) {
            this.flip_horizontal = isFlip;
            return this;
        }

        public Builder flipVertical(boolean isFlip) {
            this.flip_vertical = isFlip;
            return this;
        }

        public Builder setProjectionMethod(String method) {
            if (Arrays.asList(ZProjector.METHODS).contains(method)) {
                this.projection_method = Arrays.asList(ZProjector.METHODS).indexOf(method);
                this.is_projection = true;
            } else {
                this.is_projection = false;
            }
            return this;
        }

        public Builder setNormalization(int min, int max) {
            this.norm_min = min;
            this.norm_max = max;
            return this;
        }

        public Builder setId(File id) {
            this.id = id;
            return this;
        }

        public Builder reader(IFormatReader reader) {
            this.reader = reader;
            return this;
        }

        public Builder setRange(HyperRange range) {
            this.range = range;
            return this;
        }

        public Builder setSaveFolder(File save_folder) {
            boolean success = save_folder.mkdirs();
            if (!success) {
                if (!save_folder.exists()) {
                    throw new RuntimeException("Could not create output folder " + save_folder.getAbsolutePath());
                }
                if (!save_folder.isDirectory()) {
                    throw new RuntimeException("The destination path is not a folder (" + save_folder.getAbsolutePath() + ")");
                }
            }
            this.save_folder = save_folder;
            return this;
        }

        public Builder coordinatesCorrectionFactor(double correction_factor) {
            this.correction_factor = correction_factor;
            return this;
        }

        public Builder setDownsample(int downsample) {
            this.downsample = downsample;
            return this;
        }

        public Builder useAveraging(boolean use_averaging) {
            this.use_averaging = use_averaging;
            return this;
        }

        public Builder setTaskService(TaskService taskService) {
            this.taskService = taskService;
            return this;
        }

        public OperettaManager build() {
            if (this.id != null) {
                File id = this.id;
                if (this.id.isDirectory()) {
                    XMLFILE file = null;
                    for (XMLFILE version : XMLFILE.values()) {
                        File candidate = new File(this.id, version.getIndexFileName());
                        if (!candidate.exists()) continue;
                        file = version;
                        break;
                    }
                    if (file == null) {
                        log.error("o matching Index files found in " + this.id.getAbsolutePath());
                        log.error("Implemented valid Index files:");
                        for (XMLFILE version : XMLFILE.values()) {
                            log.error("\t" + version.getIndexFileName() + " (" + version.getDescription() + ")");
                        }
                        return null;
                    }
                    id = new File(this.id, file.getIndexFileName());
                }
                try {
                    if (this.reader == null) {
                        this.reader = Builder.createReader(id.getAbsolutePath());
                    }
                }
                catch (Exception e) {
                    log.error("Issue when creating reader for file {}", (Object)id);
                    return null;
                }
            }
            if (this.range == null) {
                this.range = new HyperRange.Builder().fromMetadata((IMetadata)this.reader.getMetadataStore()).build();
            } else if (this.range.getTotalPlanes() == 0) {
                HyperRange new_range = new HyperRange.Builder().fromMetadata((IMetadata)this.reader.getMetadataStore()).build();
                if (!this.range.getRangeC().isEmpty()) {
                    new_range.setRangeC(this.range.getRangeC());
                }
                if (!this.range.getRangeZ().isEmpty()) {
                    new_range.setRangeZ(this.range.getRangeZ());
                }
                if (!this.range.getRangeT().isEmpty()) {
                    new_range.setRangeT(this.range.getRangeT());
                }
                this.range = new_range;
            }
            if (this.save_folder == null) {
                log.warn("You did not specify a save path for the Operetta Manager object");
            }
            return new OperettaManager(this.reader, this.downsample, this.use_averaging, this.range, this.norm_min, this.norm_max, this.flip_horizontal, this.flip_vertical, this.is_projection, this.projection_method, this.save_folder, this.correction_factor, this.is_fuse_fields, this.is_use_stitcher, this.stitching_parameters, this.taskService);
        }

        public static IFormatReader createReader(String id) throws IOException, FormatException {
            log.debug("Getting new reader for " + id);
            ImageReader reader = new ImageReader();
            reader.setFlattenedResolutions(false);
            Memoizer memo = new Memoizer((IFormatReader)reader);
            IMetadata omeMetaIdxOmeXml = MetadataTools.createOMEXMLMetadata();
            memo.setMetadataStore((MetadataStore)omeMetaIdxOmeXml);
            log.debug("setId for reader " + id);
            StopWatch watch = new StopWatch();
            watch.start();
            memo.setId(id);
            watch.stop();
            log.debug("id set in " + (int)(watch.getTime() / 1000L) + " s");
            return memo;
        }
    }

    private static enum XMLFILE {
        V5("Index.idx.xml", "PerkinElmer Harmony V5"),
        V5FLEX("Index.flex.xml", "PerkinElmer Harmony V5 Flatfield data"),
        V6("Index.xml", "PerkinElmer Harmony V6");

        private final String description;
        private final String indexFileName;

        private XMLFILE(String indexFileName, String description) {
            this.indexFileName = indexFileName;
            this.description = description;
        }

        private String getIndexFileName() {
            return this.indexFileName;
        }

        public String getDescription() {
            return this.description;
        }
    }
}

